package Plugins::Qobuz::API;

use strict;
use base qw(Slim::Plugin::OPMLBased);
use JSON::XS::VersionOneAndTwo;
use LWP::UserAgent;
use URI::Escape qw(uri_escape_utf8);
use Digest::MD5 qw(md5_hex);

use constant BASE_URL => 'http://player.qobuz.com/api.json/0.2/';
use constant TRACK_URL_TTL => 600;

use Slim::Utils::Cache;
use Slim::Utils::Log;
use Slim::Utils::Prefs;

# bump the second parameter if you decide to change the schema of cached data
my $cache = Slim::Utils::Cache->new('qobuz', 1);
my $prefs = preferences('plugin.qobuz');
my $log = logger('plugin.qobuz');

my $ua = LWP::UserAgent->new(timeout => 15);

sub getToken {
	if (my $token = $cache->get('token')) {
		return $token;
	}
	
	my $result = _call('/user/login', {
		username => $prefs->get('username'),
		password => $prefs->get('password_md5_hash'),
	});
	
	if ($result && (my $token = $result->{user_auth_token})) {
		$cache->set('token', $token);
		return $token;
	}

	$cache->set('token', -1, 30);
	return;
}

sub search {
	my ($class, $search) = @_;
	
	main::DEBUGLOG && $log->debug('Search : ' . $search);
	
	return _call('search/getResults', {
		type  => 'albums',
		query => $search, 
		limit => 200,
	});
}

sub getGenres {
	my ($class, $genreId) = @_;
	
	return _call('genre/list', {
		parent_id => $genreId
	});
}

sub getGenre {
	my ($class, $genreId) = @_;
	
	return _call('genre/get', {
		genre_id => $genreId,
		extra => 'subgenresCount,albums',
	});
}

sub getAlbum {
	my ($class, $albumId) = @_;
	
	my $album = _call('album/get', {
		album_id => $albumId,
	});
	
	_precacheAlbum([$album]) if $album;
	
	return $album;
}

sub getFeaturedAlbums {
	my ($class, $type, $genreId) = @_;
	
	my $albums = _call('album/getFeatured', {
		type     => $type,
		genre_id => $genreId,
		limit    => 200,
		_ttl     => 60*60,		# features can change quite often - don't cache for too long
	});
	
	_precacheAlbum($albums->{albums}->{items}) if $albums->{albums};
	
	return $albums;
}

sub getUserPurchases {
	my ($class) = @_;
	
	my $albums = _call('purchase/getUserPurchases', {
		user_auth_token => getToken,
		limit    => 200,
		_ttl     => 2*60,		# don't cache user-controlled content for too long...
	});
	
	_precacheAlbum($albums->{albums}->{items}) if $albums->{albums};
	
	return $albums;
}

sub getUserPlaylists {
	my ($class, $user) = @_;
	
	my $playlists = _call('playlist/getUserPlaylists', {
		user_auth_token => getToken,
		username => $user || $prefs->get('username'),
		limit    => 200,
		_ttl     => 2*60,		# user playlists can change quite often - don't cache for too long
	});
	
	return $playlists;
}

sub getPublicPlaylists {
	my ($class) = @_;
	
	my $playlists = _call('playlist/getPublicPlaylists', {
		user_auth_token => getToken,
		type  => 'last-created',
		limit => 200,
		_ttl  => 60*60
	});

	return $playlists;
}

sub getPlaylistTracks {
	my ($class, $playlistId) = @_;

	my $tracks = _call('playlist/get', {
		playlist_id => $playlistId,
		extra => 'tracks',
		user_auth_token => getToken,
		_ttl  => 2*60,
	});
	
	_precachePlaylistTracks($tracks->{tracks}->{items});
	
	return $tracks;
}

sub getTrackInfo {
	my ($class, $trackId) = @_;

	if ($trackId =~ /^http/i) {
		$trackId = $cache->get("trackId_$trackId");
	}
	
	return $cache->get('trackInfo_' . $trackId) if $trackId;
}

sub getFileUrl {
	my ($class, $trackId) = @_;
	
	if (my $cached = $cache->get("trackUrl_$trackId")) {
		return $cached;
	}
	
	my $track = _call('track/getFileUrl', {
		track_id => $trackId,
		format_id => $prefs->get('preferredFormat'),
		user_auth_token => getToken,
		_ttl  => 30,
		_sign => 1,
	});
	
	if ($track->{url}) {
		# cache urls for a short time only
		$cache->set("trackUrl_$trackId", $track->{url}, TRACK_URL_TTL);
		$cache->set('trackId_' . $track->{url}, $trackId);
		return $track->{url};
	}
}

sub _precacheAlbum {
	my ($albums) = @_;
	
	foreach my $album (@$albums) { 
		my $albumInfo = {
			title  => $album->{title},
			id     => $album->{id},
			artist => $album->{artist},
			image  => $album->{image},
		};

		foreach my $track (@{$album->{tracks}->{items}}) {
			$track->{album} = $albumInfo;
			_precacheTrack($track);
		}		
	}
}

sub _precachePlaylistTracks {
	my ($tracks) = @_;
	
	foreach my $track (@$tracks) {
		_precacheTrack($track)
	}
}

sub _precacheTrack {
	my ($track) = @_;
	
warn Data::Dump::dump($track);
	$cache->set('trackInfo_' . $track->{id}, {
		title    => $track->{title},
		album    => $track->{album}->{title},
		albumId  => $track->{album}->{id},
		artist   => $track->{album}->{artist}->{name},
		artistId => $track->{album}->{artist}->{id},
		cover    => $track->{album}->{image}->{large},
	});
	
	warn Data::Dump::dump($cache->get('trackInfo_' . $track->{id}));
}

sub _call {
	my ($url, $params) = @_;
	
	$params ||= {};
	
	my @query;
	while (my ($k, $v) = each %$params) {
		next if $k =~ /^_/;		# ignore keys starting with an underscore
		push @query, $k . '=' . uri_escape_utf8($v);
	}
	
	push @query, 'app_id=942852567';
	
	# signed requests - see
	# https://github.com/Qobuz/api-documentation#signed-requests-authentification-
	if ($params->{_sign}) {
		my $signature = $url;
		$signature =~ s/\///;
		
		$signature .= join('', sort map {
			my $v = $_;
			$v =~ s/=//;
			$v;
		} grep {
			$_ !~ /(?:app_id|user_auth_token)/
		} @query);
		
		my $ts = time;
		$signature = md5_hex($signature . $ts . '761730d3f95e4af09ac63b9a37ccc96a');
		
		push @query, "request_ts=$ts", "request_sig=$signature";
		
		$params->{_nocache} = 1; 
	}
	
	$url = BASE_URL . $url . '?' . join('&', @query);
	
	main::DEBUGLOG && $log->debug($url);
	
	if (!$params->{_nocache} && (my $cached = $cache->get($url))) {
		main::DEBUGLOG && $log->debug("getting cached value");
		return $cached;
	}
	
	my $response = $ua->get($url);
	
#	main::DEBUGLOG && warn Data::Dump::dump($response);

	if ($response->is_error) {
		my $code = $response->code;
		my $message = $response->message;
		$log->error("request failed: $code  $message");
		$params->{_nocache} = 1;
	}

	my $result = eval { from_json( $response->content ) };
	main::DEBUGLOG && warn Data::Dump::dump($result);

	if (!$params->{_nocache}) {
		$cache->set($url, $result, $params->{_ttl});
	}

	return $result;
}

1;